/***************************************************************************
 *
 * Copyright (c) 2014 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/mman.h>
#include <string.h>

#include <string>
#include <fstream>
#include <sstream>

#include <wayland-client.h>
#include <wayland-client-protocol.h>

#include "ilm_client.h"
#include "ilm_control.h"
#include "ilm_types.h"
#include "WaylandServerinfoClientProtocol.h"

struct display {
    struct wl_display *display;
    struct wl_registry *registry;
    struct wl_compositor *compositor;
    struct wl_shm *shm;
    struct serverinfo *serverinfo;
    uint32_t connect_id;
};

struct window {
    struct display *display;
    int width;
    int height;
    struct wl_buffer *buffer;
    void *shm_data;
    struct wl_surface *surface;
};

static void serverinfoListener(void *data, struct serverinfo *serverinfo, uint32_t client_handle)
{
    (void)serverinfo;
    struct display *d = (struct display *)data;
    d->connect_id = client_handle;
}

struct serverinfo_listener serverinfo_listener_list = {
    serverinfoListener
};

static void shm_format(void *data, struct wl_shm* wlShm, uint32_t format)
{
    (void)data;
    (void)wlShm;
    printf("SHM format supported: 0x%x\n", format);
}

struct wl_shm_listener shm_listener = {
    shm_format
};

static void registry_handle_global(void *data, struct wl_registry *registry,
                                   uint32_t id, const char *interface,
                                   uint32_t version)
{
    (void)version;

    struct display *d = (struct display *)data;
    if (strcmp(interface, "wl_compositor") == 0)
    {
        d->compositor =
            (struct wl_compositor *)wl_registry_bind(registry, id,
                                                     &wl_compositor_interface, 1);
    }
    else if (strcmp(interface, "wl_shm") == 0)
    {
        d->shm =
            (struct wl_shm *)wl_registry_bind(registry, id, &wl_shm_interface, 1);
        wl_shm_add_listener(d->shm, &shm_listener, data);
    }
    else if (strcmp(interface, "serverinfo") == 0)
    {
        d->serverinfo =
            (struct serverinfo *)wl_registry_bind(registry, id,
                                                  &serverinfo_interface, 1);
        serverinfo_add_listener(d->serverinfo, &serverinfo_listener_list, data);
        serverinfo_get_connection_id(d->serverinfo);
    }
}

static const struct wl_registry_listener registry_listener = {
    registry_handle_global,
    NULL
};

static bool gRunning = true;

static void sigint_handler(int signum)
{
    (void)signum;
    gRunning = false;
}

static unsigned int getDepthFromFormat(enum wl_shm_format format)
{
    switch (format)
    {
        case WL_SHM_FORMAT_ARGB8888:
        case WL_SHM_FORMAT_RGBA8888:
            return 4;
        case WL_SHM_FORMAT_RGB565:
        case WL_SHM_FORMAT_UYVY:
            return 2;
        default:
            fprintf(stderr, "Cannot get depth for unknown format 0x%x\n", format);
            return 0;
    }
}

static int create_shm_file(off_t size)
{
    static const std::string path_template("/weston-shared-XXXXXX");
    const char *path = getenv("XDG_RUNTIME_DIR");

    int fd = -1;

    if (path == NULL)
    {
        fprintf(stderr, "Failed to get XDG_RUNTIME_DIR\n");
        return -1;
    }
    std::string full_template = std::string(path) + path_template;

    fd = mkstemp((char *)full_template.c_str());
    if (fd < 0)
    {
        fprintf(stderr, "mkstemp failed for path %s: %m\n", full_template.c_str());
        return -1;
    }

    if (ftruncate(fd, size) < 0)
    {
        fprintf(stderr, "ftruncate failed: %m\n");
        close(fd);
        return -1;
    }

    return fd;
}

static void destroy_window(struct window *window)
{
    wl_surface_destroy(window->surface);
    wl_buffer_destroy(window->buffer);
}

static struct window * create_window(struct display* display, int width,
                                     int height, enum wl_shm_format format)
{
    struct window *window;
    struct wl_shm_pool *pool;
    void *data;
    int stride, size, fd;
    unsigned int depth;

    if (display == NULL)
    {
        fprintf(stderr, "display is NULL in create_window\n");
        return NULL;
    }

    window = (struct window *)calloc(1, sizeof *window);
    if (window == NULL)
    {
        fprintf(stderr, "Failed to allocate memory for window\n");
        return NULL;
    }

    window->display = display;
    window->width = width;
    window->height = height;
    window->surface = wl_compositor_create_surface(display->compositor);
    if (window->surface == NULL)
    {
        fprintf(stderr, "Failed to create surface from compositor\n");
        goto fail;
    }

    depth = getDepthFromFormat(format);
    if (depth == 0)
    {
        fprintf(stderr, "Failed to get depth for format 0x%x\n", format);
        goto fail;
    }

    stride = width * depth;
    size = stride * height;

    fd = create_shm_file(size);
    if (fd < 0)
    {
        fprintf(stderr, "Creating a buffer file of size %d B failed: %m\n",
                size);
    }

    data = mmap(NULL, size, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
    if (data == MAP_FAILED)
    {
        fprintf(stderr, "mmap failed: %m\n");
        goto fail_fd;
    }

    pool = wl_shm_create_pool(display->shm, fd, size);
    if (pool == NULL)
    {
        fprintf(stderr, "Failed to create pool: %m\n");
        goto fail_fd;
    }

    window->buffer = wl_shm_pool_create_buffer(pool, 0, width, height,
                                               stride, format);
    if (window->buffer == NULL)
    {
        fprintf(stderr, "Failed to create SHM buffer: %m\n");
        goto fail_pool;
    }

    wl_shm_pool_destroy(pool);
    close(fd);

    window->shm_data = data;

    return window;

    fail_pool:
        wl_shm_pool_destroy(pool);
    fail_fd:
        close(fd);
    fail:
        free(window);
        return NULL;
}

static void destroy_display(struct display *display)
{
    if (display->shm)
        wl_shm_destroy(display->shm);

    if (display->compositor)
        wl_compositor_destroy(display->compositor);

    wl_registry_destroy(display->registry);
    wl_display_flush(display->display);
    wl_display_disconnect(display->display);
    free(display);
}

static struct display* create_display()
{
    struct display *display;
    display = (struct display *)malloc(sizeof *display);
    if (display == NULL)
    {
        fprintf(stderr, "Failed to allocate memory for display\n");
        return NULL;
    }
    display->display = wl_display_connect(NULL);
    if (display->display == NULL)
    {
        fprintf(stderr, "display->display is NULL\n");
        goto cleanup;
    }

    display->registry = wl_display_get_registry(display->display);
    if (display->registry == NULL)
    {
        fprintf(stderr, "display->registry is NULL\n");
        goto cleanup;
    }

    wl_registry_add_listener(display->registry, &registry_listener, display);

    wl_display_roundtrip(display->display);
    if (display->shm == NULL)
    {
        fprintf(stderr, "display->shm is NULL\n");
        goto cleanup;
    }

    wl_display_roundtrip(display->display); //TODO: Understand why wl_display_roundtrip is called twice

    return display;

    cleanup:
        free(display);
        return NULL;
}

uint32_t RGBtoY(uint32_t r, uint32_t g, uint32_t b)
{
    return ((66 * r + 129 * g + 25 * b + 128) >> 8) + 16;
}

static void RGBtoYUV(uint32_t r, uint32_t g, uint32_t b,
                     uint32_t &y, uint32_t &u, uint32_t &v)
{
    y = RGBtoY(r, g, b);
    u = ((-38 * r - 74 * g + 112 * b + 128) >> 8) + 128;
    v = ((112 * r - 94 * g - 18 * b + 128) >> 8) + 128;
}

static void getrgba(int x, int y, int width, int height,
                    unsigned int &red, unsigned int &green,
                    unsigned int &blue, unsigned int &alpha)
{
    const int halfh = height / 2;
    const int halfw = width / 2;
    int inner_radius, outer_radius;

    // squared radii thresholds
    outer_radius = (halfw < halfh ? halfw : halfh) - 8;
    inner_radius = outer_radius - 32;
    outer_radius *= outer_radius;
    inner_radius *= inner_radius;

    // squared distance from the centre   
    int r2 = (x - halfw) * (x - halfw) + (y - halfh) * (y - halfh);
    if (r2 < inner_radius)
    {
        red = (r2 * 8 / 32) % 256;
        green = (r2 * 4 / 32) % 256;
        blue = (r2 / 32) % 256;
    }
    else if (r2 < outer_radius)
    {
        red = (y * 8) % 256;
        green = (y * 4) % 256;
        blue = y % 256;
    }
    else
    {
        red = (x * 8) % 256;
        green = (x * 4) % 256;
        blue = x % 256;
    }

    /* draw a cross using the alpha channel */
    if (abs(x - y) > 6 && abs(x + y - height) > 6)
        alpha = 255;
    else
        alpha = 0;
}

static bool paint_pixels(void *image, int width, int height,
                         enum wl_shm_format format)
{
    bool retcode = true;
    char *pixel = (char *)image;
    unsigned int red = 0, green = 0, blue = 0, alpha = 0;
    int y;

    for (y = 0; (y < height && retcode == true); y++)
    {
        int x;
        for (x = 0; (x < width && retcode == true); x++)
        {

            if (format == WL_SHM_FORMAT_RGBA8888)
            {
                // stuff the RGBA into a 32-bit word.
                uint32_t value = 0x00000000;
                getrgba(x, y, width, height, red, green, blue, alpha);
                value |= alpha;
                value |= (blue << 8);
                value |= (green << 16);
                value |= (red << 24);
                *(uint32_t *)pixel = value;
                pixel += 4;
            }
            else if (format == WL_SHM_FORMAT_ARGB8888)
            {
                // stuff the ARGB into a 32-bit word.
                uint32_t value = 0x00000000;
                getrgba(x, y, width, height, red, green, blue, alpha);
                value |= blue;
                value |= (green << 8);
                value |= (red << 16);
                value |= (alpha << 24);
                *(uint32_t *)pixel = value;
                pixel += 4;
            }
            else if (format == WL_SHM_FORMAT_RGB565)
            {
                // Stuff this coordinate into a 16-bit word
                uint16_t value = 0x0000;
                getrgba(x, y, width, height, red, green, blue, alpha);
                red >>= (8 - 5);
                green >>= (8 - 6);
                blue >>= (8 - 5);
                value |= blue;
                value |= (green << 5);
                value |= (red << 11);
                *(uint16_t *)pixel = value;
                pixel += 2;
            }
            else if (format == WL_SHM_FORMAT_UYVY)
            {
                // Stuff two pixels into a 32-bit word
                uint32_t value = 0x00000000;
                uint32_t U, Y1, V, Y2;
                getrgba(x, y, width, height, red, green, blue, alpha);
                RGBtoYUV(red, green, blue, Y1, U, V);

                // Manually advance to the next coordinate
                x++;
                if (x == width)
                {
                    x = 0;
                    y++;
                }

                getrgba(x, y, width, height, red, green, blue, alpha);
                Y2 = RGBtoY(red, green, blue);
                value |= Y2;
                value |= (V << 8);
                value |= (Y1 << 16);
                value |= (U << 24);
                *(uint32_t *)pixel = value;
                pixel += 4;
            }
            else
            {
                fprintf(stderr, "Unknown format 0x%x\n", format);
                retcode = false;
            }
        }
    }

    return retcode;
}

static bool dump_padded(void *buffer, unsigned int width, unsigned int height,
                        std::string& path, enum wl_shm_format format,
                        unsigned int lstrip, unsigned int rstrip,
                        unsigned int tstrip, unsigned int bstrip,
                        unsigned int lpad, unsigned int rpad,
                        unsigned int tpad, unsigned int bpad)
{
    // TODO: Consider if depth varies
    static const char zero = 0;
    unsigned int depth;
    bool retval = true;

    if (!(depth = getDepthFromFormat(format)))
    {
        fprintf(stderr, "Cannot dump, failed to get format\n");
        return false;
    }

    unsigned int total_width = width + lpad + rpad - lstrip - rstrip;
    off_t top_skip = tstrip * width * depth;
    off_t left_skip = lstrip * depth;
    off_t right_skip = rstrip * depth;
    size_t width_len = width * depth - left_skip - right_skip;

    char *pos = (char *)buffer + top_skip + left_skip;

    std::ofstream outfile(path.c_str(), std::ofstream::binary);
    if (!outfile.good())
    {
        fprintf(stderr, "Failed to open file %s\n", path.c_str());
        retval = false;
    }
    else
    {
        // write top padding
        for (unsigned int i = 0; i < total_width * tpad * depth; i++)
            outfile.write(&zero, 1);

        // write each line for the buffer
        for (unsigned int i = 0;
             i < height - bstrip - tstrip;
             i++, pos += width * depth)
        {
            // write left padding
            for (unsigned int j = 0; j < lpad * depth; j++)
                outfile.write(&zero, 1);

            // write line (minus rstrip and lstrip)
            outfile.write(pos, width_len);

            // write right padding
            for (unsigned int j = 0; j < rpad * depth; j++)
                outfile.write(&zero, 1);
        }

        // write bottom padding
        for (unsigned int i = 0; i < total_width * bpad * depth; i++)
            outfile.write(&zero, 1);

        if (outfile.fail())
        {
            fprintf(stderr, "Error occurred while writing to output file %s\n",
                    path.c_str());
            retval = false;
        }
        outfile.close();
    }
    return retval;
}

static void * create_output_buffer(unsigned int width, unsigned int height,
                                   enum wl_shm_format format)
{
    void *ptr = malloc(width * height * getDepthFromFormat(format));
    if (ptr == NULL)
    {
        fprintf(stderr, "Failed to allocate memory for buffer\n");
        return NULL;
    }
    paint_pixels(ptr, width, height, format);
    return ptr;
}

int main(int argc, char **argv)
{
    int exit_code = 0;
    int ret = 0;
    ilmErrorTypes error = ILM_FAILED;

    t_ilm_surface surfaceID = 2355;
    t_ilm_layer layerID = 2450;
    int width = 1024;
    int height = 768;
    enum wl_shm_format wl_format = WL_SHM_FORMAT_ARGB8888;
    ilmPixelFormat lm_format = ILM_PIXELFORMAT_RGBA_8888;

    enum wl_shm_format output_format = WL_SHM_FORMAT_RGBA8888;
    void *output_buffer;

    struct display *display;
    struct window *window;
    struct wl_proxy* pxy;
    uint32_t id;
    t_ilm_nativehandle native_ilm_handle;

    struct sigaction sigint;

    std::string dump_path;
    unsigned int lstrip = 0, rstrip = 0, tstrip = 0, bstrip = 0;
    unsigned int lpad = 0, rpad = 0, tpad = 0, bpad = 0;

    // parse args

    for (int i=1; i < argc; i++)
    {
        char *argval = strchr(argv[i], '=') + 1;
        std::string argvalstr(argval);
        std::istringstream iss(argvalstr);

        if (strncmp("-path=", argv[i], sizeof("-path=") - 1) == 0)
            dump_path = argvalstr;
        else if (strncmp("-width=", argv[i], sizeof("-width=") -1) == 0)
            iss >> width;
        else if (strncmp("-height=", argv[i], sizeof("-height=") -1) == 0)
            iss >> height;
        else if (strncmp("-surface=", argv[i], sizeof("-surface=") -1) == 0)
            iss >> surfaceID;
        else if (strncmp("-layer=", argv[i], sizeof("-layer=") -1) == 0)
            iss >> layerID;
        else if (strncmp("-lstrip=", argv[i], sizeof("-lstrip=") - 1) == 0)
            iss >> lstrip;
        else if (strncmp("-rstrip=", argv[i], sizeof("-rstrip=") - 1) == 0)
            iss >> rstrip;
        else if (strncmp("-tstrip=", argv[i], sizeof("-tstrip=") - 1) == 0)
            iss >> tstrip;
        else if (strncmp("-bstrip=", argv[i], sizeof("-bstrip=") - 1) == 0)
            iss >> bstrip;
        else if (strncmp("-lpad=", argv[i], sizeof("-lpad=") - 1) == 0)
            iss >> lpad;
        else if (strncmp("-rpad=", argv[i], sizeof("-rpad=") - 1) == 0)
            iss >> rpad;
        else if (strncmp("-tpad=", argv[i], sizeof("-tpad=") - 1) == 0)
            iss >> tpad;
        else if (strncmp("-bpad=", argv[i], sizeof("-bpad=") - 1) == 0)
            iss >> bpad;
        else if (strncmp("-format=", argv[i], sizeof("-format=") - 1) == 0)
        {
            // TODO: Intelligently report unsupported formats
            if (argvalstr.compare("RGBA8888") == 0)
            {
                wl_format = WL_SHM_FORMAT_RGBA8888;
                lm_format = ILM_PIXELFORMAT_RGBA_8888;

                fprintf(stderr, "Error: This pixel format is not supported\n");
                exit_code = -1;
                goto exit;
            }
            else if (argvalstr.compare("ARGB8888") == 0)
            {
                wl_format = WL_SHM_FORMAT_ARGB8888;

                // NOTE: No equivalent ilmPixelFormat exists
                lm_format = ILM_PIXELFORMAT_RGBA_8888;
            }
            else if (argvalstr.compare("RGB565") == 0)
            {
                wl_format = WL_SHM_FORMAT_RGB565;
                lm_format = ILM_PIXELFORMAT_RGB_565;

                fprintf(stderr, "Error: This pixel format is not supported\n");
                exit_code = -1;
                goto exit;
            }
            else if (argvalstr.compare("UYVY") == 0)
            {
                wl_format = WL_SHM_FORMAT_UYVY;
                lm_format = ILM_PIXELFORMAT_UYVY;

                fprintf(stderr, "Error: This pixel format is not supported\n");
                exit_code = -1;
                goto exit;
            }
            else
            {
                fprintf(stderr, "Invalid format\n");
                exit_code = -1;
                goto exit;
            }
        }
        else if (strncmp("-output-format=", argv[i], sizeof("-output-format=") - 1) == 0)
        {
            if (argvalstr.compare("RGBA8888") == 0)
            {
                output_format = WL_SHM_FORMAT_RGBA8888;
            }
            else if (argvalstr.compare("ARGB8888") == 0)
            {
                output_format = WL_SHM_FORMAT_ARGB8888;
            }
            else if (argvalstr.compare("RGB565") == 0)
            {
                output_format = WL_SHM_FORMAT_RGB565;
            }
            else if (argvalstr.compare("UYVY") == 0)
            {
                output_format = WL_SHM_FORMAT_UYVY;
            }
            else
            {
                fprintf(stderr, "Invalid format\n");
                exit_code = -1;
                goto exit;
            }
        }
    }

    if (argc <= 1)
    {

        fprintf(stderr,
                "Usage: %s -path=DUMP_PATH [OPTIONS]\n"
                "Where options are:\n"
                "  -lstrip=[LEFT_STRIP]\n"
                "  -rstrip=[RIGHT_STRIP]\n"
                "  -tstrip=[TOP_STRIP]\n"
                "  -bstrip=[BOTTOM_STRIP]\n"
                "  -lpad=[LEFT_PADDING]\n"
                "  -rpad=[RIGHT_PADDING]\n"
                "  -tpad=[TOP_PADDING]\n"
                "  -bpad=[BOTTOM_PADDING]\n"
                "  -format=[FORMAT]\n"
                "  -output-format=[FORMAT]\n"
                "Where output formats are:\n"
                "  ARGB8888\n"
                "  RGBA8888\n"
                "  RGB565\n"
                "  UYVY\n",
                argv[0]);
        exit_code = -1;
        goto exit;
    }

    if (dump_path.size() == 0)
    {
        fprintf(stderr, "No output file path is set\n");
        exit_code = -1;
        goto exit;
    }

    // Set the signal handler
    sigint.sa_handler = sigint_handler;
    sigemptyset(&sigint.sa_mask);
    sigint.sa_flags = SA_RESETHAND;
    sigaction(SIGINT, &sigint, NULL);
    sigaction(SIGTERM, &sigint, NULL);

    // Initialize things
    error = ilm_init();
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to create surface: %s\n",
                ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit;
    }

    display = create_display();
    if (display == NULL)
    {
        fprintf(stderr, "Failed to create display\n");
        exit_code = 1;
        goto exit_ilm;
    }

    window = create_window(display, width, height, wl_format);
    if (window == NULL)
    {
        fprintf(stderr, "Failed to create window\n");
        exit_code = -1;
        goto exit_display;
    }

    // Set various bits of layermanager information

    pxy = (struct wl_proxy *)window->surface;
    id = (uint32_t) wl_proxy_get_id(pxy);
    native_ilm_handle = (display->connect_id << 16) | id;

    error = ilm_surfaceCreate(native_ilm_handle, width, height, lm_format, &surfaceID);
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to create surface: %s\n",
                ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit_window;
    }

    error = ilm_surfaceSetDestinationRectangle(surfaceID, 0, 0, width, height);
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to set surface destination rectangle: %s\n",
                ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit_surface;
    }

    error = ilm_surfaceSetSourceRectangle(surfaceID, 0, 0, width, height);
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to set surface source rectangle: %s\n",
                ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit_surface;
    }

    error = ilm_surfaceSetVisibility(surfaceID, ILM_TRUE); // see ilm_control.h
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to set surface opacity: %s\n",
                ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit_surface;
    }

    error = ilm_layerAddSurface(layerID, surfaceID);
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to add surface %d to layer %d: %s\n",
                surfaceID, layerID, ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit_surface;
    }

    error = ilm_commitChanges();
    if (error != ILM_SUCCESS)
    {
        fprintf(stderr, "Failed to commit changes: %s\n",
                ILM_ERROR_STRING(error));
        exit_code = -1;
        goto exit_surface;
    }

    // Paint once
    printf("Painting pixels\n");
    if (!paint_pixels(window->shm_data, width, height, wl_format))
    {
        fprintf(stderr, "Failed to paint pixels\n");
        exit_code = -1;
        goto exit_surface;
    }
    wl_surface_attach(window->surface, window->buffer, 0, 0);
    wl_surface_damage(window->surface, 0, 0, window->width, window->height);
    wl_surface_commit(window->surface);

    // Write a buffer to file
    printf("Dumping to %s\n", dump_path.c_str());

    output_buffer = create_output_buffer(width, height, output_format);

    if (!dump_padded(output_buffer, width, height, dump_path, output_format,
                     lstrip, rstrip, tstrip, bstrip,
                     lpad, rpad, tpad, bpad))
    {
        fprintf(stderr, "Failed to dump window\n");
        exit_code = -1;
        goto exit_output_buffer;
    }

    // Enter run loop
    printf("Entering run loop\n");
    while (gRunning && ret != -1)
        ret = wl_display_dispatch(display->display);



    exit_output_buffer:
        free(output_buffer);
    exit_surface:
        error = ilm_surfaceRemove(surfaceID);
        if (error != ILM_SUCCESS)
        {
            fprintf(stderr, "Failed to remove surface: %s\n",
                    ILM_ERROR_STRING(error));
            exit_code = -1;
        }

        error = ilm_commitChanges();
        if (error != ILM_SUCCESS)
        {
            fprintf(stderr, "Failed to commit changes: %s\n",
                    ILM_ERROR_STRING(error));
            exit_code = -1;
        }
    exit_window:
        destroy_window(window);
    exit_display:
        destroy_display(display);
    exit_ilm:
        ilm_destroy();
    exit:
        return exit_code;
}
